import { createI18n } from 'vue-i18n'
import en from './locales/en.json'
import zh from './locales/zh.json'

// List of supported languages
const availableLocales = ['en', 'zh']
  
// Auto-detect user language preference
const browserLocale = (navigator.language || navigator.userLanguage).substring(0, 2)
const defaultLocale = availableLocales.includes(browserLocale) ? browserLocale : 'en'

// Message object
const messages = {
  en,
  zh
}

// Create i18n instance
const i18n = createI18n({
  legacy: false, // Use Composition API mode
  locale: defaultLocale,
  fallbackLocale: 'en', // Fallback language
  messages,
  globalInjection: true, // Globally inject $t function
  allowDynamic: true, // Allow dynamic addition of translations
  silentFallbackWarn: true, // Silent fallback warnings
  missingWarn: false, // Disable missing translation warnings
  fallbackWarn: false, // Disable fallback warnings
})

// Optional: Add global translation function prototype extension
i18n.global.setLocaleMessage('en', en)
i18n.global.setLocaleMessage('zh', zh)

// Reactive language switching function
export const setLanguage = (locale) => {
  if (availableLocales.includes(locale)) {
    i18n.global.locale = locale
    // Optional: Save language preference to localStorage
    localStorage.setItem('locale', locale)
    return true
  }
  return false
}

// Get current language
export const getCurrentLanguage = () => {
  return i18n.global.locale.value
}

// Add new translation packages
export const addTranslations = (locale, messages) => {
  if (!i18n.global.availableLocales.includes(locale)) {
    i18n.global.setLocaleMessage(locale, messages)
    i18n.global.locale.value = locale
  }
  return i18n.global.availableLocales.includes(locale)
}

export default i18n

// Language switching for Vue panels
export const availableLanguages = [
  { code: 'en', name: 'English', flag: '🇺🇸' },
  { code: 'zh', name: 'Chinese', flag: '🇨🇳' }
]

// Language switching composition function
export function useLanguageSwitcher() {
  const { locale } = useI18n()
  
  const switchLanguage = (newLocale) => {
    if (availableLocales.includes(newLocale)) {
      locale.value = newLocale
      localStorage.setItem('locale', newLocale)
      return true
    }
    return false
  }
  
  const getCurrentLanguageInfo = () => {
    return availableLanguages.find(lang => lang.code === locale.value) || availableLanguages[0]
  }
  
  return {
    switchLanguage,
    getCurrentLanguageInfo,
    currentLocale: locale,
    availableLocales
  }
}
